module output_aerocom_aie 
!------------------------------------------------------------------------------------------
! 
!  Purpose:  to output variables requried by AEROCOM AIE intercomparison. 
! 
!  Method:  to sample cloud properties at the cloud top to facilliate the comparions 
!           between the model and satellite observations. 
!           Also to sample cloud properties only at the time of satellite overpass. 
!
!  Author: Minghuai Wang (2008-04)
! 
!  Updated by Minghuai Wang on August, 2013 for the third AeroCOM AIE intercomparison (IND3)
!
!------------------------------------------------------------------------------------------  

   use shr_kind_mod, only: r8 => shr_kind_r8
   use ppgrid,       only: pcols, pver, begchunk, endchunk
   use cam_history,      only: addfld, add_default,  outfld, horiz_only
   use cam_history_support,   only : fillvalue
   use cam_abortutils,    only: endrun
   use crmdims,         only: crm_nx, crm_ny, crm_nz

   implicit none

   private 

   public ::  output_aerocom_aie_init 
   public ::  output_aerocom_aie_register
   public ::  aerocom_calc
   public :: cloud_top_aerocom
   public ::  output_aerocom

   logical, public :: do_aerocom_ind3 = .true.

   real(r8) :: p0 = 1.0e5    

!  variable index for pbuf
   integer :: crm_rel_idx      ! droplet effective radius at CRM scales
   integer :: crm_rei_idx      ! ice crystal effective radius at CRM scales
   integer :: crm_autoconv_idx      ! Column-integrated autoconversion rate at CRM scales
   integer :: crm_accretn_idx      ! Column-integrated autoconversion rate at CRM scales
   integer :: crm_aod400_idx    ! 
   integer :: aod400_idx    ! 
   integer :: aod700_idx    ! 
   integer :: crm_aod700_idx    ! 
   integer :: crm_tau_idx    ! 

   integer :: cdr_idx      ! droplet effective radius at top of liquid water clouds (meter)
   integer :: cdnc_idx     ! droplet number concentration in top layer of liquid water clouds (#/m3)
   integer :: cdnum_idx    ! column-integrated droplet number concentrations
   integer :: icnum_idx    ! column-integrated ice crystal number concentrations
   integer :: clt_idx      ! fractional cover by all clouds
   integer :: lcc_idx      ! fractional cover by liquid water clouds
   integer :: lwp_idx      ! in-cloud liquid water path for liquid clouds (kg/m^2)
   integer :: iwp_idx      !  in-cloud ice water path for ice clouds (kg/m^2)
   integer :: icr_idx      ! effective radius of crystals at top of ice clouds (meter)
   integer :: icc_idx      ! fractional cover by ice clouds
   integer :: cod_idx      ! in-cloud optical depth 
   integer :: ccn_idx      ! cloud condensation nuclei number concentration for liquid water
                                       ! clouds where activation corresponding to CDR and CDN
   integer :: ttop_idx     ! Temperature at top of clouds (K)
   integer :: ptop_idx     ! Pressure at top fo clouds (Pa)
   integer :: autoconv_idx      ! Column-integrated autoconversion rate
   integer :: accretn_idx      ! Column-integrated autoconversion rate
   integer :: rh700_idx      ! relative humidity at 700 hPa
   integer :: icnc_idx     ! ice crystal number concentration in top layer of ice clouds (#/m3)

   integer :: intccn_idx   
   integer :: colrv_idx
   integer :: rwp_idx
   integer :: lwp2_idx
   integer :: iwp2_idx

   integer :: crm_ccn3d_idx    ! 

   integer :: cldo_idx     ! pbuf index cloud fraction 
   integer :: qrain_idx    ! pbuf index for grid-mean rain water
   integer :: cld_tau_idx  ! cloud optical depth
   integer :: rel_idx      ! droplet effective radius
   integer :: rei_idx      ! ice crystal effective radius
   
   integer :: cldliqbf_idx  ! liquid water before microphyiscs
   integer :: cldicebf_idx  ! ice water before microphyiscs
   integer :: numliqbf_idx  ! liquid droplet number before microphyiscs
   integer :: numicebf_idx  ! ice crystal number before microphyiscs

   integer :: ixcldice, ixcldliq, ixnumliq, ixnumice

CONTAINS

   subroutine  output_aerocom_aie_register ()
   use physics_buffer,      only : pbuf_add_field, dtype_r8
    
   call pbuf_add_field('crm_aod400',   'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny/),          crm_aod400_idx )
   call pbuf_add_field('crm_aod700',   'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny/),          crm_aod700_idx )
   call pbuf_add_field('aod400',   'physpkg', dtype_r8, (/pcols/),          aod400_idx )
   call pbuf_add_field('aod700',   'physpkg', dtype_r8, (/pcols/),          aod700_idx )
   call pbuf_add_field('TAU_crm',   'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny, crm_nz/),          crm_tau_idx )
   call pbuf_add_field('REL_crm',   'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny, crm_nz/),          crm_rel_idx )
   call pbuf_add_field('REI_crm',   'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny, crm_nz/),          crm_rei_idx )
   call pbuf_add_field('CCN3_crm',   'physpkg', dtype_r8, (/pcols, crm_nx, crm_ny, crm_nz/),          crm_ccn3d_idx )
   call pbuf_add_field('autoconv_crm', 'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny, crm_nz/),      crm_autoconv_idx)
   call pbuf_add_field('accretn_crm', 'physpkg', dtype_r8, (/pcols,crm_nx, crm_ny, crm_nz/),      crm_accretn_idx)



   return
   end subroutine output_aerocom_aie_register
!----------------------------------------------------------------------------------------------------

!====================================================================================================
   subroutine  output_aerocom_aie_init ()
   use infnan
   use physics_buffer,      only : pbuf_get_index, dtype_r8
   use constituents,  only: cnst_get_ind

  call addfld ('angstrm',      horiz_only,           'A', '#', 'Angstrom coefficient', flag_xyfill=.true.) 
  call addfld ('angstrm_crm', (/'crm_nx','crm_ny'/),           'A', '#', 'Angstrom coefficient at CRM grids', flag_xyfill=.true.) 
  call addfld ('cdr',    horiz_only, 'A','micro meter',   'GCM-cell mean droplet effective radius at top of liquid water clouds',  flag_xyfill=.true.)
   call addfld ('cdr_crm',    (/'crm_nx','crm_ny'/), 'A','micro meter',   'CRM-cell mean droplet effective radius at top of liquid water clouds',  flag_xyfill=.true.)
   call addfld ('cdnc',    horiz_only,'A', '#/m3',    'Grid-cell mean droplet number concentration at top of liquid water clouds',  flag_xyfill=.true.)
   call addfld ('cdnc_crm',    (/'crm_nx','crm_ny'/),'A', '#/m3',    'CRM-cell mean droplet number concentration at top of liquid water clouds',  flag_xyfill=.true.)
   call addfld ('cod',     horiz_only, 'A', 'amount',  'GCM-cell mean cloud optical depth',  flag_xyfill=.true.)
   call addfld ('cod_crm',     (/'crm_nx','crm_ny'/),  'A', 'amount',  'CRM-cell mean cloud optical depth',  flag_xyfill=.true.)
   call addfld ('cdnum_crm',    (/'crm_nx','crm_ny'/), 'A','#/m2',   'CRM-cell mean column-integrated droplet number concentrations',  flag_xyfill=.true.)
   call addfld ('cdnum',   horiz_only, 'A','#/m2',   'Grid-cell mean column-integrated droplet number concentrations',  flag_xyfill=.true.)
   call addfld ('icnum',   horiz_only, 'A', '#/m2',    'Grid-cell mean column-integrated ice crystal number concentrations', flag_xyfill=.true.)
   call addfld ('icnum_crm',   (/'crm_nx','crm_ny'/), 'A', '#/m2',    'CRM-cell mean column-integrated ice crystal number concentrations', flag_xyfill=.true.)
   call addfld ('clt',   horiz_only, 'A', 'fraction','Fractional cover by all clouds',   flag_xyfill=.true.) 
   call addfld ('lcc',   horiz_only, 'A', 'fraction','Fractional cover by liquid water clouds',   flag_xyfill=.true.)
   call addfld ('icc',   horiz_only, 'A', 'fraction','Fractional cover by ice clouds',  flag_xyfill=.true.)
   call addfld ('lwp',   horiz_only, 'A', 'kg/m2',   'Grid-cell mean liquid water path for liquid water clouds', flag_xyfill=.true.)
   call addfld ('lwp_crm',   (/'crm_nx','crm_ny'/), 'A', 'kg/m2',   'CRM-cell mean liquid water path for liquid water clouds', flag_xyfill=.true.)
   call addfld ('iwp',   horiz_only, 'A', 'kg/m2',   'Grid-cell mean ice water path for ice clouds',  flag_xyfill=.true.)
   call addfld ('iwp_crm',   (/'crm_nx','crm_ny'/), 'A', 'kg/m2',   'CRM-cell mean ice water path for liquid water clouds', flag_xyfill=.true.)
   call addfld ('icnc',  horiz_only, 'A', '#/m3', 'Ice crystal number concentration at top of ice clouds', flag_xyfill=.true.)
   call addfld ('icnc_crm',  (/'crm_nx','crm_ny'/), 'A', '#/m3', 'Ice crystal number concentration at top of ice clouds at CRM scales', flag_xyfill=.true.)
   call addfld ('icr',    horiz_only, 'A', 'meter',   'Grid-cell mean effective radius of crystals at top of ice clouds', flag_xyfill=.true.)
   call addfld ('icr_crm',    (/'crm_nx','crm_ny'/), 'A', 'meter',   'CRM-cell mean effective radius of crystals at top of ice clouds', flag_xyfill=.true.)
   call addfld ('rh700', horiz_only, 'A', 'fraction',  'Relative humidity at 700 hPa',  flag_xyfill=.true.)
   call addfld ('ccn',  horiz_only, 'A',  '#/m3',    'CCN number concentration at 0.3% at the top layer of liquid water clouds',  flag_xyfill=.true.)
   call addfld ('rwp',  horiz_only, 'A','kg/m2',  'Rain water path',  flag_xyfill=.true.)
   call addfld ('rwp_crm',  (/'crm_nx','crm_ny'/), 'A','kg/m2',  'CRM-cell Rain water path',  flag_xyfill=.true.)
   call addfld ('ttop',   horiz_only, 'A','K',       'Temperature at top of clouds',  flag_xyfill=.true.)
   call addfld ('ptop',   horiz_only, 'A','Pa',       'Pressure at top of clouds',  flag_xyfill=.true.)
   call addfld ('ttop_crm',    (/'crm_nx','crm_ny'/), 'A','K',       'Temperature at top of clouds at CRM-scale',  flag_xyfill=.true.)
   call addfld ('ptop_crm',    (/'crm_nx','crm_ny'/), 'A','Pa',       'Pressure at top of cloudsat CRM-scale ',  flag_xyfill=.true.)
   call addfld ('autoconv', horiz_only, 'A',  'kg/m2/s',       'Column-integrated auto-conversion rate', flag_xyfill=.true.)
   call addfld ('accretn',   horiz_only, 'A', 'kg/m2/s',       'Column-integrated accretion rate', flag_xyfill=.true.)
   call addfld ('autoconv_crm', (/'crm_nx','crm_ny'/), 'A',  'kg/m2/s',       'Column-integrated auto-conversion rate', flag_xyfill=.true.)
   call addfld ('accretn_crm',  (/'crm_nx','crm_ny'/) , 'A', 'kg/m2/s',       'Column-integrated accretion rate', flag_xyfill=.true.)
   call addfld ('colrv',  horiz_only, 'A','micrometer * kg/m2',  'Column-integrated volume-mean droplet effective radius',  flag_xyfill=.true.)
   call addfld ('lwp2',   horiz_only, 'A', 'kg/m2',   'Grid-cell mean liquid water path for liquid water clouds (new)',  flag_xyfill=.true.)
   call addfld ('iwp2',   horiz_only, 'A', 'kg/m2',   'Grid-cell mean ice water path for liquid water clouds (new)',  flag_xyfill=.true.)
   call addfld ('colrv_crm',  (/'crm_nx','crm_ny'/), 'A','micrometer * kg/m2',  'Column-integrated volume-mean droplet effective radius at CRM scale',  flag_xyfill=.true.)
   call addfld ('lwp2_crm',   (/'crm_nx','crm_ny'/), 'A', 'kg/m2',   'CRM-cell mean liquid water path for liquid water clouds (new)',  flag_xyfill=.true.)
   call addfld ('iwp2_crm',   (/'crm_nx','crm_ny'/), 'A', 'kg/m2',   'CRM-cell mean ice water path for liquid water clouds (new)',  flag_xyfill=.true.)
   call addfld('ccn.1bl',horiz_only, 'A','#/m3   ','CCN concentration at S=0.1% at 1km above surface')
   call addfld('ccn.3bl',horiz_only, 'A','#/m3   ','CCN concentration at S=0.3% at 1km above surface')


 


   return
   end subroutine output_aerocom_aie_init
!=================================================================================

!---------------------------------------------------------------------------------
   subroutine cloud_top_aerocom(state, pbuf)
!-----------------------------------------------------------------------------------
!  Purpose: to sample the variables at the cloud top for both ice and liquid clouds. 
!
!  Source: the original codes is  provided by AEROCOM (AIE intercomparison)
!
!------------------------------------------------------------------------------------

   use physics_types,   only: physics_state
   use physics_buffer, only : physics_buffer_desc, pbuf_get_field, pbuf_get_index
   use physconst,     only: gravit, rair, cappa

   type(physics_state), intent(in), target :: state
   type(physics_buffer_desc), pointer      :: pbuf(:)
   

!  Local variables

  real(r8) :: crm_cdr(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_cdnc(pcols,crm_nx,crm_ny) 
  real(r8) :: colcdnc(pcols) 
  real(r8) :: crm_colcdnc(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_colicnc(pcols,crm_nx,crm_ny) 
  real(r8) :: tcc(pcols) 
  real(r8) :: lcc(pcols) 
  real(r8) :: icc(pcols) 
  real(r8) :: crm_lwp(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_lwp2(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_iwp2(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_iwp(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_icnc(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_ccn(pcols,crm_nx,crm_ny) 
  !real :: crm_ccn(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_ptop(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_ttop(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_rwp(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_colacret(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_colauto(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_colrv(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_icr(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_cod(pcols,crm_nx,crm_ny) 
  real(r8) :: crm_ange(pcols,crm_nx,crm_ny) 
  integer :: cloud_flag(pcols,crm_nx,crm_ny) 
  integer :: liquid_flag(pcols,crm_nx,crm_ny) 
  integer :: ice_flag(pcols,crm_nx,crm_ny) 
  real(r8) :: rhoair(pcols, pver)   ! air density (kg/m3)
   real(r8) :: airmass (pcols, pver) 
   real(r8) :: zaltitude (pcols, pver)
   real(r8) :: dz(pcols, pver)
   real(r8) :: lwp2(pcols)
   real(r8) :: iwp2(pcols)
   real(r8) :: lwp(pcols)
   real(r8) :: rel(pcols)
   real(r8) :: rei(pcols)
   real(r8) :: tau(pcols)
   real(r8) :: ttop(pcols)
   real(r8) :: ptop(pcols)
   real(r8) :: cdr(pcols)
   real(r8) :: icr(pcols)
   real(r8) :: cdnc(pcols)
   real(r8) :: icnc(pcols)
   real(r8) :: ccn(pcols)
   real(r8) :: iwp(pcols)
   real(r8) :: cdnum(pcols)
   real(r8) :: icnum(pcols)
   real(r8) :: auto(pcols)
   real(r8) :: acret(pcols)
   real(r8) :: colrv(pcols)
   real(r8) :: colicnc(pcols)
   real(r8) :: ange(pcols)
   real(r8) :: rwp(pcols)
   real(r8) :: cod(pcols)

   real(r8), pointer, dimension(:, :, :) :: crm_aod400 
   real(r8), pointer, dimension(:, :, :) :: crm_aod700 
   real(r8), pointer, dimension(: ) :: aod400 
   real(r8), pointer, dimension(: ) :: aod700 
   real(r8), pointer, dimension(:, :, :, :) :: crm_rel
   real(r8), pointer, dimension(:, :, :, :) :: crm_rei
   real(r8), pointer, dimension(:,:,:,:)   :: crm_nc
   real(r8), pointer, dimension(:,:,:,:)   :: crm_ni
   real(r8), pointer, dimension(:,:,:,:)   :: crm_qc
   real(r8), pointer, dimension(:,:,:,:)   :: crm_qi
   real(r8), pointer, dimension(:,:,:,:)   :: crm_qr
   real(r8), pointer, dimension(:,:,:,:)   :: crm_tau
   real(r8), pointer, dimension(:,:,:,:)   :: crm_tabs
   real(r8), pointer, dimension(:,:,:,:)   :: crm_acret
   real(r8), pointer, dimension(:,:,:,:)   :: crm_auto
   !real(r8), pointer, dimension(:,:,:,:)   :: ccn3d 
   real, pointer, dimension(:,:,:,:)   :: ccn3d 


   real(r8) :: ftmp(pcols)
   real(r8) :: zi(pver+1) 
   real(r8) :: flag_max 
   real(r8) :: thres_cld, thres_cod, thres_cwp, max_cld, fr, cwp
   real(r8) :: rv, t700
   real(r8) :: iiwp, ilwp, iphase, lphase
 
   integer :: icrmx, icrmy, icrmz, il, nlev 
   integer :: ncrm_z, ncrm_x, ncrm_y 
   integer :: itrue
   integer :: lchnk, ncol
   integer :: i, k
   integer ::  crm_nc_idx,  crm_ni_idx, crm_qc_idx, crm_qi_idx, &
              crm_qr_idx, crm_tabs_idx
   
   thres_cld = 0.001
   thres_cwp = 1.0e-5  
   max_cld = 1.0_r8 - 1.0e-9
!   thres_cod = 0.3
   thres_cod = 1.0e-5 
   nlev = pver
   lchnk = state%lchnk
   ncol  = state%ncol

   rhoair(:ncol, :) = state%pmid(:ncol,:) / (rair * state%t(:ncol,:))

    crm_aod400_idx      = pbuf_get_index('crm_aod400') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_aod400_idx, crm_aod400)
    crm_aod700_idx      = pbuf_get_index('crm_aod700') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_aod700_idx, crm_aod700)
    aod400_idx      = pbuf_get_index('aod400') !Guangxing Lin
   call pbuf_get_field(pbuf, aod400_idx, aod400)
    aod700_idx      = pbuf_get_index('aod700') !Guangxing Lin
   call pbuf_get_field(pbuf, aod700_idx, aod700)
    crm_rel_idx      = pbuf_get_index('REL_crm') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_rel_idx,  crm_rel) !Guangxing Lin
    crm_nc_idx      = pbuf_get_index('CRM_NC') !Guangxing Lin
   call pbuf_get_field (pbuf, crm_nc_idx,        crm_nc)   
    crm_ni_idx      = pbuf_get_index('CRM_NI') !Guangxing Lin
   call pbuf_get_field (pbuf, crm_ni_idx,        crm_ni)   
    crm_tau_idx      = pbuf_get_index('TAU_crm') !Guangxing Lin
   call pbuf_get_field (pbuf, crm_tau_idx,        crm_tau)   
    crm_qc_idx      = pbuf_get_index('CRM_QC') !Guangxing Lin
   call pbuf_get_field (pbuf, crm_qc_idx,        crm_qc)   
    crm_qi_idx      = pbuf_get_index('CRM_QI') !Guangxing Lin
   call pbuf_get_field (pbuf, crm_qi_idx,        crm_qi)   
    crm_rei_idx      = pbuf_get_index('REI_crm') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_rei_idx,  crm_rei) !Guangxing Lin
    !ccn3d_idx      = pbuf_get_index('CCN3') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_ccn3d_idx, ccn3d)
    crm_qr_idx      = pbuf_get_index('CRM_QR') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_qr_idx,  crm_qr) !Guangxing Lin
    crm_accretn_idx      = pbuf_get_index('accretn_crm') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_accretn_idx,  crm_acret) !Guangxing Lin
    crm_autoconv_idx      = pbuf_get_index('autoconv_crm') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_autoconv_idx,  crm_auto) !Guangxing Lin
    crm_tabs_idx      = pbuf_get_index('CRM_T') !Guangxing Lin
   call pbuf_get_field(pbuf, crm_tabs_idx,  crm_tabs) !Guangxing Lin
   
  zaltitude = 0.0
   do i=1, ncol
     zi = 0.0
     do k=pver, 1, -1
       airmass(i,k) = state%pdel(i,k)/gravit     ! kg/m2
       dz(i, k) = airmass(i,k)/rhoair(i,k) ! layer thickness in m
       zi(k) = zi(k+1) + dz(i,k) ! layer thickness in m
       zaltitude(i, k) = (zi(k+1)+zi(k))/2._r8
     end do
   end do
   
   icc(:) = 0.0_r8
   lcc(:) = 0.0_r8
   ttop(:) = 0.0_r8
   ptop(:) = 0.0_r8
   cdr(:) = 0.0_r8
   icr(:) = 0.0_r8
   cdnc(:) = 0.0_r8
   icnc(:) = 0.0_r8
   ccn(:) = 0.0_r8
   lwp2(:) = 0.0_r8
   iwp2(:) = 0.0_r8
   
   auto =0.0_r8
   acret =0.0_r8

    cdnc=0.0_r8;  
     colcdnc=0.0_r8 
   cod=0.0_r8; lwp=0.0_r8; iwp=0.0_r8
   rwp=0.0_r8 
   tcc = 0.0_r8; 
  
   liquid_flag =0
   ice_flag =0
   cloud_flag =0

   crm_ttop = 0.0_r8
   crm_ptop = 0.0_r8
   crm_cdr=0.0_r8; crm_cdnc=0.0_r8; crm_ccn=0.0_r8; 
   colcdnc=0.0_r8; crm_colicnc=0.0_r8
   crm_colrv=0.0_r8;
   crm_icr=0.0_r8; 
   crm_cod=0.0_r8;  crm_lwp=0.0_r8; crm_iwp=0.0_r8
   crm_lwp2=0.0_r8 
   crm_iwp2=0.0_r8 
   crm_rwp=0.0_r8; 
   crm_colauto = 0.0_r8; crm_colacret = 0.0_r8
   crm_ange = 0.0_r8

   ncrm_z= crm_nz
   ncrm_x= crm_nx
   ncrm_y = crm_ny

   do i=1, ncol
     do icrmx=1, crm_nx
      do icrmy=1, crm_ny
        do icrmz=crm_nz, 1, -1
         il=nlev-icrmz+1 
         if( crm_qc(i,icrmx,icrmy,icrmz).gt.5.0e-5.or.(crm_qi(i,icrmx,icrmy,icrmz).gt.1.0e-6) ) then !  not too small cloud 
            cloud_flag (i,icrmx,icrmy) = 1
            crm_ttop(i,icrmx,icrmy) = crm_tabs(i,icrmx,icrmy,icrmz)
            crm_ptop(i,icrmx,icrmy) = state%pmid(i,il) 
            ttop(i) = ttop(i) + crm_tabs(i,icrmx,icrmy,icrmz)
            ptop(i) = ptop(i) + state%pmid(i,il) 
            tcc(i) = tcc(i)+1.0
            if(crm_qc(i,icrmx,icrmy,icrmz).gt.5.0e-5) then  ! 5.0e-5 is used in CAM5
              crm_cdr(i,icrmx,icrmy) = crm_rel(i,icrmx,icrmy,icrmz)
              cdr(i) = cdr(i) + crm_rel(i,icrmx,icrmy,icrmz)
              crm_cdnc(i,icrmx,icrmy) = crm_nc(i,icrmx,icrmy,icrmz) * rhoair(i,il)
              cdnc(i) = cdnc(i) + crm_nc(i,icrmx,icrmy,icrmz) * rhoair(i,il)
              crm_ccn(i, icrmx,icrmy) = ccn3d(i,icrmx,icrmy,icrmz)
              liquid_flag(i,icrmx,icrmy) = 1
              ccn(i) = ccn(i) + ccn3d(i,icrmx,icrmy,icrmz)
              lcc(i) = lcc(i) + 1.0
            end if
           
            if(crm_qi(i,icrmx,icrmy,icrmz).gt.1.0e-6) then   ! 1.0e-6 is used in CAM5
              crm_icr(i,icrmx,icrmy) = crm_rei(i,icrmx,icrmy,icrmz)
              crm_icnc(i,icrmx,icrmy) = crm_ni(i,icrmx,icrmy,icrmz) * rhoair(i,il)
              ice_flag(i,icrmx,icrmy) = 1
              icr(i) = icr(i) + crm_rei(i,icrmx,icrmy,icrmz)
              icnc(i) = icnc(i) + crm_ni(i,icrmx,icrmy,icrmz) * rhoair(i,il)
              icc(i) = icc(i) + 1.0
            end if
            exit
          end if ! is there a visible, not-too-small cloud?
        end do  ! end icrmz

        if(liquid_flag(i,icrmx,icrmy).eq.0) then
          crm_cdr(i,icrmx,icrmy) = fillvalue
          crm_cdnc(i,icrmx,icrmy) = fillvalue
          
          crm_ccn(i,icrmx,icrmy) = fillvalue
          crm_lwp2(i,icrmx,icrmy) = fillvalue
        else
           do  icrmz=1, ncrm_z
            il=nlev-icrmz+1
            crm_lwp2(i,icrmx,icrmy) = crm_lwp2(i,icrmx,icrmy)+                                &
              crm_qc(i,icrmx,icrmy, icrmz) * airmass(i,il)  ! kg/kg --> kg/m2
          end do !icrmz 
            lwp2(i)=lwp2(i) + crm_lwp2(i,icrmx,icrmy)
        end if !liquid_flag

        if(ice_flag(i,icrmx,icrmy).eq.0) then
          crm_icr(i,icrmx,icrmy) = fillvalue
          crm_icnc(i,icrmx,icrmy) = fillvalue
          crm_iwp2(i,icrmx,icrmy) = fillvalue
        else
           do  icrmz=1, ncrm_z
            il=nlev-icrmz+1
            crm_iwp2(i,icrmx,icrmy) = crm_iwp2(i,icrmx,icrmy)+                                &
              crm_qi(i,icrmx,icrmy, icrmz) * airmass(i,il)  ! kg/kg --> kg/m2
          end do !icrmz 
            iwp2(i)=iwp2(i) + crm_iwp2(i,icrmx,icrmy)

        end if !ice flag

         if(cloud_flag(i,icrmx,icrmy).eq.0) then
          crm_ttop(i,icrmx,icrmy) = fillvalue
          crm_ptop(i,icrmx,icrmy) = fillvalue
        else
!          print *,  'test cod', crm_cod(i,icrmx,icrmy) , crm_cod2(i,icrmx,icrmy)
        end if !ice flag

       


! calculate Angstrom exponent at CRM scales
        if(crm_aod400(i,icrmx,icrmy).lt.1.0e4 .and. crm_aod700(i,icrmx,icrmy).lt.1.e4  .and. &
           crm_aod400(i,icrmx,icrmy).gt.1.0e-10 .and. crm_aod700(i,icrmx, icrmy).gt.1.0e-10) then
          crm_ange(i,icrmx,icrmy) = log (crm_aod400(i,icrmx,icrmy))-log(crm_aod700(i,icrmx,icrmy))
          crm_ange(i,icrmx,icrmy) = crm_ange(i,icrmx,icrmy)/(log(0.700_r8)-log(0.400_r8))
        else
          crm_ange(i,icrmx,icrmy) = fillvalue
        end if

!do column integeration at CRM scales
        do  icrmz=1, ncrm_z
            il=nlev-icrmz+1
            crm_cod(i,icrmx,icrmy) = crm_cod(i,icrmx,icrmy)+crm_tau(i,icrmx,icrmy,icrmz)
            crm_colcdnc(i,icrmx,icrmy)=crm_colcdnc(i,icrmx,icrmy)+    &
                 crm_nc(i,icrmx,icrmy,icrmz) * airmass(i,il) ! #/m2 
            crm_colicnc(i,icrmx,icrmy)=crm_colicnc(i,icrmx,icrmy)+    &
                 crm_ni(i,icrmx,icrmy,icrmz) * airmass(i,il) ! #/m2 
            crm_lwp(i,icrmx,icrmy) = crm_lwp(i,icrmx,icrmy)+                                &
              crm_qc(i,icrmx,icrmy, icrmz) * airmass(i,il)   !  kg/m2
            crm_iwp(i,icrmx,icrmy) = crm_iwp(i,icrmx,icrmy)+                                &
                crm_qi(i,icrmx,icrmy,icrmz) * airmass(i,il)   !  kg/m2
            crm_rwp(i,icrmx,icrmy) = crm_rwp(i,icrmx,icrmy)+                                &
                crm_qr(i,icrmx,icrmy,icrmz) * airmass(i,il)   !  kg/m2
            crm_colacret(i,icrmx,icrmy) = crm_colacret(i,icrmx,icrmy)+          &
                crm_acret(i,icrmx,icrmy,icrmz) * airmass(i,il)   !  kg/m2
            crm_colauto(i,icrmx,icrmy) = crm_colauto(i,icrmx,icrmy)+          &
                crm_auto(i,icrmx,icrmy,icrmz) * airmass(i,il)   !  kg/m2
       !if(crm_auto(i,icrmx,icrmy,icrmz).gt.1e10.or.crm_acret(i,icrmx,icrmy,icrmz).gt.1e10) then
       !  print *, 'gxlin-test3, auto= ', crm_auto(i,icrmx,icrmy,icrmz), 'acret= ', crm_acret(i,icrmx,icrmy,icrmz)
       !end if

             rv = min(25._r8, max(1.0_r8, (3 * crm_qc(i,icrmx,icrmy,icrmz)/max(4*3.14159*crm_nc(i,icrmx,icrmy,icrmz)*1.0e3, 1.0_r8))**(1/3.) * 1.0e6))  ! micrometer
            crm_colrv(i,icrmx,icrmy) = crm_colrv(i,icrmx,icrmy)+  &
                rv * crm_qc(i,icrmx,icrmy,icrmz) * airmass(i,il)  
        end do !ncrm_z

! do sum up over the GCM grid           
          cod(i) = cod(i) + crm_cod(i,icrmx,icrmy)
          colcdnc(i)=colcdnc(i)+crm_colcdnc(i,icrmx,icrmy)
          colicnc(i)=colicnc(i)+crm_colicnc(i,icrmx,icrmy)
          lwp(i)=lwp(i) + crm_lwp(i,icrmx,icrmy)
          iwp(i)=iwp(i) + crm_iwp(i,icrmx,icrmy)
          rwp(i)=rwp(i) + crm_rwp(i,icrmx,icrmy)
          colrv(i)=colrv(i) + crm_colrv(i,icrmx,icrmy)
          acret(i)=acret(i) + crm_colacret(i,icrmx,icrmy)
          auto(i)=auto(i) + crm_colauto(i,icrmx,icrmy)
       end do !icrmx 
     end do !icrmy

     if(tcc(i).gt.0.5) then
       ttop(i) = ttop(i)/tcc(i)
       ptop(i) = ptop(i)/tcc(i)
       !cod(i) = cod(i)/tcc(i)
       tcc(i) = tcc(i)/(ncrm_x*ncrm_y)
     else
       ttop(i) = fillvalue
       ptop(i) = fillvalue
       !cod(i) = fillvalue
     end if
     if(lcc(i).gt.0.5) then
       ccn(i) = ccn(i)/lcc(i)
       cdr(i) = cdr(i)/lcc(i)
       cdnc(i) = cdnc(i)/lcc(i)
       lwp2(i) = lwp2(i)/lcc(i)
       lcc(i) = lcc(i)/(ncrm_x*ncrm_y)
     else
       ccn(i) = fillvalue
       cdr(i) = fillvalue
       cdnc(i) = fillvalue
       lwp2(i) = fillvalue
     end if
     if(icc(i).gt.0.5) then
       icr(i) = icr(i)/icc(i)
       icnc(i) = icnc(i)/icc(i)
       icc(i) = icc(i)/(ncrm_x*ncrm_y)
       iwp2(i) = iwp2(i)/icc(i)

     else
        icr(i) = fillvalue
       icnc(i) = fillvalue
       iwp2(i) = fillvalue

     end if
! average over the GCM grid

         cod(i) = cod(i)/(crm_nx*crm_ny)
         colcdnc(i)=colcdnc(i)/(crm_nx*crm_ny)
         colicnc(i)=colicnc(i)/(crm_nx*crm_ny)
         iwp(i)=iwp(i)/(crm_nx*crm_ny)
         lwp(i)=lwp(i)/(crm_nx*crm_ny)
         rwp(i) = rwp(i) / (ncrm_x*ncrm_y)
         colrv(i) = colrv(i) / (ncrm_x*ncrm_y)
         acret(i) = acret(i) / (ncrm_x*ncrm_y)
         auto(i) = auto(i) / (ncrm_x*ncrm_y)
        
       !if(auto(i).gt.1e10.or.acret(i).gt.1e10) then
       !  print *, 'gxlin-test2, auto= ', auto(i), 'acret= ', acret(i)
       !end if
 
     if(aod400(i).lt.1.0e4 .and. aod700(i).lt.1.e4  .and. &
        aod400(i).gt.1.0e-10 .and. aod700(i).gt.1.0e-10) then
          ange(i) = log (aod400(i))-log(aod700(i))
          ange(i) = ange(i)/(log(0.700)-log(0.400))
     else
          ange(i) = fillvalue
     end if

   end do ! loop over I

   cdnum = colcdnc

   call outfld('angstrm_crm', crm_ange, pcols, lchnk) 
   call outfld('angstrm', ange, pcols, lchnk) 
   call outfld('cdr_crm', crm_cdr, pcols, lchnk) 
   call outfld('cdr', cdr, pcols, lchnk) 
   call outfld('cdnc_crm', crm_cdnc, pcols, lchnk)
   call outfld('cdnc', cdnc, pcols, lchnk)
   call outfld('cod_crm', crm_cod, pcols, lchnk)
   call outfld('cod', cod, pcols, lchnk)
   call outfld('cdnum', cdnum, pcols, lchnk)
   call outfld('cdnum_crm', crm_colcdnc, pcols, lchnk)
   call outfld('icnum', colicnc, pcols, lchnk)
   call outfld('icnum_crm', crm_colicnc, pcols, lchnk)
   call outfld('clt', tcc, pcols, lchnk)
   call outfld('icc', icc, pcols, lchnk)
   call outfld('lcc', lcc, pcols, lchnk)
   call outfld('lwp', lwp, pcols, lchnk)
   call outfld('iwp', iwp, pcols, lchnk)
   call outfld('lwp_crm', crm_lwp, pcols, lchnk)
   call outfld('iwp_crm', crm_iwp, pcols, lchnk)
   call outfld('icnc', icnc, pcols, lchnk)
   call outfld('icnc_crm', crm_icnc, pcols, lchnk)
   call outfld('icr', icr, pcols, lchnk)
   call outfld('icr_crm', crm_icr, pcols, lchnk)
   call outfld('ccn', ccn, pcols, lchnk)
   call outfld('ptop', ptop, pcols, lchnk)
   call outfld('ttop', ttop, pcols, lchnk)
   call outfld('ttop_crm', crm_ttop, pcols, lchnk)
   call outfld('ptop_crm', crm_ptop, pcols, lchnk)
   call outfld('rwp', rwp, pcols, lchnk)
   call outfld('rwp_crm', crm_rwp, pcols, lchnk)
   call outfld('colrv', colrv, pcols, lchnk)
   call outfld('lwp2', lwp2, pcols, lchnk)
   call outfld('iwp2', iwp2, pcols, lchnk)
   call outfld('colrv_crm', crm_colrv, pcols, lchnk)
   call outfld('lwp2_crm', crm_lwp2, pcols, lchnk)
   call outfld('iwp2_crm', crm_iwp2, pcols, lchnk)
   !call outfld('autoconv', auto, pcols, lchnk)
   !call outfld('accretn', acret, pcols, lchnk)
   !call outfld('autoconv_crm', crm_colauto, pcols, lchnk)
   !call outfld('accretn_crm', crm_colacret, pcols, lchnk)
  

 

   return
   end subroutine cloud_top_aerocom
!==========================================================================

!--------------------------------------------------------------------------
   subroutine aerocom_calc(state, cld, rel, rei, lnd, ind, tau, cliqwp, cicewp, coszrs)
!-------------------------------------------------------------------
!  calculate required variables.
!------------------------------------------------------------------
   use physics_types,   only: physics_state

   implicit none
!!#include <comctl.h>

   type(physics_state), intent(in) :: state
   real(r8), intent(in) :: cld(pcols,pver)     ! cloud cover
   real(r8), intent(in) :: rel(pcols,pver)     ! Liquid cloud particle effective radius (microns)
   real(r8), intent(in) :: rei(pcols,pver)     ! Ice effective drop size (microns) 
   real(r8), intent(in) :: lnd(pcols, pver)    ! Liquid cloud number concentration (#/cm3)
   real(r8), intent(in) :: ind(pcols, pver)    ! Liquid cloud number concentration (#/cm3)
   real(r8), intent(in) :: tau(pcols, pver)    ! cloud optical depth 
   real(r8), intent(in) :: cliqwp(pcols,pver)  ! in-cloud liquid water path  (g/m2)
   real(r8), intent(in) :: cicewp(pcols,pver)  ! in-cloud ice water path (g/m2)
   real(r8), intent(in) :: coszrs(pcols)       ! cosine solar zenith angle (to tell if day or night)

!  Local variables
   integer :: lchnk, ncol
   integer :: i, k

   lchnk = state%lchnk
   ncol  = state%ncol

!   call cloud_top_aerocom(state, pbuf)

   return
   end subroutine aerocom_calc
!-----------------------------------------------------------------------------

!=============================================================================
   subroutine output_aerocom (state, pbuf)
   use physics_types,   only: physics_state
   use physics_buffer, only : physics_buffer_desc, pbuf_get_field

   type(physics_state), intent(in), target :: state
   type(physics_buffer_desc), pointer      :: pbuf(:)

!  Local variables
   
!   if(dosw) then
!     call outfld('OD550',   od550(:),   pcols, lchnk)
!     call outfld('ALBS',    albs(:),    pcols, lchnk)
!     call outfld('RST',     rst(:),     pcols, lchnk)
!     call outfld('RSTCS',   rstcs(:),   pcols, lchnk)
!     call outfld('RSS',     rss(:),     pcols, lchnk)
!     call outfld('RSSCS',   rsscs(:),     pcols, lchnk)
!     call outfld('RSDS',    rsds(:),    pcols, lchnk)
!   end if
!   if (dolw) then
!     call outfld('RLT',     rlt(:),     pcols, lchnk)
!     call outfld('RLTCS',   rltcs(:),   pcols, lchnk)
!     call outfld('RLS',     rls(:),     pcols, lchnk)
!     call outfld('RLSCS',     rlscs(:),     pcols, lchnk)
!   end if

!   call outfld('HFLS',    hfls(:),    pcols, lchnk)
!   call outfld('HFSS',    hfss(:),    pcols, lchnk)

   return
   end subroutine output_aerocom
!--------------------------------------------------------------------------

end module  output_aerocom_aie 
