module EDCLMLinkMod

  ! ============================================================================
  ! Modules to control the passing of infomation generated by ED into CLM to be used for either 
  ! diagnostics, or as input to the land surface components. 
  ! ============================================================================

  use shr_kind_mod          , only : r8 => shr_kind_r8;
  use decompMod             , only : bounds_type
  use clm_varpar            , only : nclmax, nlevcan_ed, numpft, numcft
  use clm_varctl            , only : iulog 
  use landunit_varcon       , only : istsoil
  use CNCarbonFluxType      , only : carbonflux_type
  use CNCarbonStateType     , only : carbonstate_type
  use CNNitrogenStateType   , only : nitrogenstate_type
  use CanopyStateType       , only : canopystate_type
  use WaterStateType        , only : waterstate_type
  use EcophysConType        , only : ecophyscon
  use PatchType             , only : pft
  use ColumnType            , only : col
  use LandunitType          , only : lun
  use EDVecPatchtype        , only : EDpft
  use EDBioType             , only : EDbio_type
  use EDEcophysConType      , only : EDecophyscon
  use EDPhysiologyMod       , only : root_fraction
  use EDtypesMod            , only : site, patch, cohort, gridcell_edstate_type
  use EDtypesMod            , only : area, dinc_ed, hitemax, numpft_ed, n_hite_bins

  implicit none
  save
  private

  public :: clm_ed_link
  public :: clm_indices
  public :: clm_leaf_area_profile
  public :: update_ed_history_variables

  logical :: DEBUG = .false.  ! for debugging this module (EDCLMLinkMod.F90)

  ! ============================================================================
  ! 10/30/09: Created by Rosie Fisher
  ! ============================================================================

contains

  ! ============================================================================
  subroutine clm_ed_link( bounds, geds_local, &
       waterstate_vars, canopystate_vars, EDbio_vars, &
       carbonstate_vars, nitrogenstate_vars, carbonflux_vars) 

    use EDGrowthFunctionsMod  , only: tree_lai, c_area

    implicit none
    type(bounds_type)           , intent(in)            :: bounds  ! clump bounds
    type(gridcell_edstate_type) , intent(inout), target :: geds_local( bounds%begg: )
    type(waterstate_type)       , intent(inout)         :: waterstate_vars
    type(canopystate_type)      , intent(inout)         :: canopystate_vars
    type(EDbio_type)            , intent(inout)         :: EDbio_vars
    type(carbonstate_type)      , intent(inout)         :: carbonstate_vars
    type(nitrogenstate_type)    , intent(inout)         :: nitrogenstate_vars
    type(carbonflux_type)       , intent(inout)         :: carbonflux_vars

    type (site)   , pointer :: currentSite
    type (patch)  , pointer :: currentPatch
    type (cohort) , pointer :: currentCohort

    integer  :: g,p,c
    integer  :: ft                                      ! plant functional type
    integer  :: patchn                                  ! identification number for each patch. 
    integer  :: firstsoilpatch(bounds%begg:bounds%endg) ! the first patch in this gridcell that is soil and thus bare... 
    real(r8) :: total_bare_ground                       ! sum of the bare fraction in all pfts.
    real(r8) :: total_patch_area                               
    real(r8) :: coarse_wood_frac  
    real(r8) :: canopy_leaf_area                        ! total amount of leaf area in the vegetated area. m2.  
    integer  :: sitecolumn(bounds%begg:bounds%endg)
    
    associate(& 
         ED_patch      => EDpft%ED_patch              , &
         ED_bareground => EDpft%ED_bareground         , &

         tlai          => canopystate_vars%tlai_patch , &
         elai          => canopystate_vars%elai_patch , &
         tsai          => canopystate_vars%tsai_patch , &
         esai          => canopystate_vars%esai_patch , &
         htop          => canopystate_vars%htop_patch , &
         hbot          => canopystate_vars%hbot_patch   & 
         )

      if (DEBUG) then
         write(iulog,*) 'in ed clm link'
      endif

      call clm_indices( bounds, geds_local )

      ! retrieve the first soil patch associated with each gridcell. 
      firstsoilpatch(bounds%begg:bounds%endg) = -999
      do c = bounds%begc,bounds%endc
         g = col%gridcell(c)
         ! make sure we only get the first patch value for places which have soil. 
         if( (lun%itype(col%landunit(c)) == istsoil).and.(col%itype(c) == istsoil))then 
            firstsoilpatch(g) = col%pfti(c)
            sitecolumn(g) = c
         endif
      enddo

      ! ============================================================================
      ! Zero the whole variable so we dont have ghost values when patch number declines.
      ! ============================================================================

      ED_patch(bounds%begp:bounds%endp)      = 0 
      ED_bareground(bounds%begp:bounds%endp) = 0 
      tlai(bounds%begp:bounds%endp)          = 0.0_r8    
      elai(firstsoilpatch(g))                = 0.0_r8
      tsai(firstsoilpatch(g))                = 0.0_r8
      esai(firstsoilpatch(g))                = 0.0_r8
      htop(bounds%begp:bounds%endp)          = 0.0_r8      
      hbot(bounds%begp:bounds%endp)          = 0.0_r8   

      do g = bounds%begg,bounds%endg
         currentSite => geds_local(g)%spnt
         if(firstsoilpatch(g) >= 0.and.currentSite%istheresoil == 1)then 
            currentSite%clmcolumn = sitecolumn(g)

            ! ============================================================================
            ! Zero the bare ground tile BGC variables.
            ! ============================================================================

            tlai(firstsoilpatch(g)) = 0.0_r8    
            htop(firstsoilpatch(g)) = 0.0_r8      
            hbot(firstsoilpatch(g)) = 0.0_r8      

            patchn = 0
            total_bare_ground = 0.0_r8
            total_patch_area = 0._r8 

            currentPatch => currentSite%oldest_patch
            do while(associated(currentPatch))
               patchn = patchn + 1
               currentPatch%patchno = patchn

               if(patchn <= numpft - numcft)then !don't expand into crop patches.   
                  currentPatch%clm_pno = firstsoilpatch(g) + patchn !the first 'soil' patch is unvegetated...      
                  p = currentPatch%clm_pno
                  c = pft%column(p)

                  call root_fraction(currentPatch)
                  ED_patch(p) = 1 !this .is. a tile filled with vegetation... 

                  !zero cohort-summed variables. 
                  currentPatch%total_canopy_area  = 0.0_r8
                  currentPatch%total_tree_area    = 0.0_r8
                  currentPatch%lai                = 0.0_r8
                  canopy_leaf_area                = 0.0_r8

                  !update cohort quantitie s                                  
                  currentCohort => currentPatch%shortest
                  do while(associated(currentCohort))
                     ft = currentCohort%pft
                     currentCohort%livestemn = currentCohort%bsw  / ecophyscon%leafcn(currentCohort%pft)

                     if (ecophyscon%woody(ft) == 1) then
                        coarse_wood_frac = 0.5_r8
                     else
                        coarse_wood_frac = 0.0_r8
                     end if

                     currentCohort%livecrootn = currentCohort%br * coarse_wood_frac / ecophyscon%leafcn(ft)
                     currentCohort%b          = currentCohort%balive+currentCohort%bdead+currentCohort%bstore
                     currentCohort%treelai    = tree_lai(currentCohort)
                     ! Why is currentCohort%c_area used and then reset in the
                     ! following line?
                     canopy_leaf_area         = canopy_leaf_area + currentCohort%treelai *currentCohort%c_area
                     currentCohort%c_area     = c_area(currentCohort)

                     if(currentCohort%canopy_layer.eq.1)then
                        currentPatch%total_canopy_area = currentPatch%total_canopy_area + currentCohort%c_area
                        if(ecophyscon%woody(ft).eq.1)then
                           currentPatch%total_tree_area = currentPatch%total_tree_area + currentCohort%c_area
                        endif
                     endif

                     ! Check for erroneous zero values. 
                     if(currentCohort%dbh <= 0._r8.or.currentCohort%n == 0._r8)then
                        write(iulog,*) 'ED: dbh or n is zero in clmedlink', currentCohort%dbh,currentCohort%n
                     endif
                     if(currentCohort%pft == 0.or.currentCohort%canopy_trim <= 0._r8)then
                       write(iulog,*) 'ED: PFT or trim is zero in clmedlink',currentCohort%pft,currentCohort%canopy_trim
                     endif
                     if(currentCohort%balive <= 0._r8)then
                       write(iulog,*) 'ED: balive is zero in clmedlink',currentCohort%balive
                     endif

                     currentCohort => currentCohort%taller

                  enddo ! ends 'do while(associated(currentCohort))

                  if ( currentPatch%total_canopy_area-currentPatch%area > 0.000001_r8 ) then
                     write(iulog,*) 'canopy area bigger than area',currentPatch%total_canopy_area ,currentPatch%area
                     currentPatch%total_canopy_area = currentPatch%area
                  endif

                  ! PASS BACK PATCH-LEVEL QUANTITIES THAT ARE NEEDED BY THE CLM CODE
                  if (associated(currentPatch%tallest)) then
                     htop(p) = currentPatch%tallest%hite
                  else
                     ! FIX(RF,040113) - should this be a parameter for the minimum possible vegetation height?
                     htop(p) = 0.1_r8
                  endif

                  hbot(p) = max(0._r8, min(0.2_r8, htop(p)- 1.0_r8))

                  ! leaf area index: of .only. the areas with some vegetation on them, as the non-vegetated areas 
                  ! are merged into the bare ground fraction. This introduces a degree of unrealism, 
                  ! which could be fixed if the surface albedo routine took account of the possibiltiy of bare 
                  ! ground mixed with trees. 

                  if(currentPatch%total_canopy_area > 0)then;
                     tlai(p) = canopy_leaf_area/currentPatch%total_canopy_area 
                  else
                     tlai(p) = 0.0_r8
                  endif

                  !write(iulog,*) 'tlai',tlai(p)
                  !write(iulog,*) 'htop',htop(p)

                  ! We are assuming here that grass is all located underneath tree canopies. 
                  ! The alternative is to assume it is all spatial distinct from tree canopies.
                  ! In which case, the bare area would have to be reduced by the grass area...
                  ! currentPatch%total_canopy_area/currentPatch%area is fraction of this patch cover by plants 
                  ! currentPatch%area/AREA is the fraction of the soil covered by this patch. 
                  EDpft%wtED(p) = min(1.0_r8,(currentPatch%total_canopy_area/currentPatch%area)) * (currentPatch%area/AREA)
                  currentPatch%bare_frac_area = (1.0_r8 - min(1.0_r8,currentPatch%total_canopy_area/currentPatch%area)) * &
                       (currentPatch%area/AREA)                 
                  ! write(iulog,*) 'bare frac',currentPatch%bare_frac_area
                  total_patch_area = total_patch_area + EDpft%wtED(p) + currentPatch%bare_frac_area
                  total_bare_ground = total_bare_ground + currentPatch%bare_frac_area
                  currentCohort=> currentPatch%tallest

               else
                  write(iulog,*) 'ED: too many patches' 
               end if ! patchn<15

               currentPatch => currentPatch%younger
            end do !patch loop

            if((total_patch_area-1.0_r8).gt.1e-9)then
               write(iulog,*) 'total area is wrong in CLMEDLINK',total_patch_area
            endif

            !loop round all and zero the remaining empty vegetation patches 
            do p = firstsoilpatch(g)+patchn+1,firstsoilpatch(g)+numpft   
               EDpft%wtED(p)    = 0.0_r8
            enddo

            !set the area of the bare ground patch. 
            p = firstsoilpatch(g) 
            EDpft%wtED(p)       = total_bare_ground
            ED_bareground(p)  = 1
         endif ! are there any soil patches?    

         call clm_leaf_area_profile(currentSite, waterstate_vars, canopystate_vars ) 

      end do !grid loop

      call update_ed_history_variables( bounds, geds_local, firstsoilpatch, EDbio_vars, &
           carbonstate_vars, nitrogenstate_vars, carbonflux_vars, &
           canopystate_vars )

    end associate

  end subroutine clm_ed_link

  ! ============================================================================
  subroutine update_ed_history_variables( bounds, geds_local , firstsoilpatch, EDbio_vars, &
       carbonstate_vars, nitrogenstate_vars, carbonflux_vars, canopystate_vars ) 

    use filterMod

    implicit none

    type(bounds_type)           , intent(in)            :: bounds  ! clump bounds
    type(gridcell_edstate_type) , intent(inout), target :: geds_local( bounds%begg: )
    type (site)                 , pointer               :: currentSite
    type (patch)                , pointer               :: currentPatch
    type (cohort)               , pointer               :: currentCohort
    type(EDbio_type)            , intent(inout)         :: EDbio_vars
    type(carbonstate_type)      , intent(inout)         :: carbonstate_vars
    type(nitrogenstate_type)    , intent(inout)         :: nitrogenstate_vars
    type(carbonflux_type)       , intent(inout)         :: carbonflux_vars
    type(canopystate_type)      , intent(inout)         :: canopystate_vars

    integer  :: G,p,ft
    integer  :: firstsoilpatch(bounds%begg:bounds%endg)
    real(r8) :: n_density   ! individual of cohort per m2.

    associate(&
         trimming                       => EDbio_vars%trimming_patch             , & ! Output:
         canopy_spread                  => EDbio_vars%canopy_spread_patch        , & ! Output:
         PFTbiomass                     => EDbio_vars%PFTbiomass_patch           , & ! Output:
         PFTleafbiomass                 => EDbio_vars%PFTleafbiomass_patch       , & ! Output:
         PFTstorebiomass                => EDbio_vars%PFTstorebiomass_patch      , & ! Output:
         PFTnindivs                     => EDbio_vars%PFTnindivs_patch           , & ! Output:
         area_plant                     => EDbio_vars%area_plant_patch           , & ! Output:
         area_trees                     => EDbio_vars%area_trees_patch           , & ! Output:
         nesterov_fire_danger           => EDbio_vars%nesterov_fire_danger_patch , & ! Output:
         spitfire_ROS                   => EDbio_vars%spitfire_ROS_patch         , & ! Output:
         effect_wspeed                  => EDbio_vars%effect_wspeed_patch        , & ! Output:
         TFC_ROS                        => EDbio_vars%TFC_ROS_patch              , & ! Output:
         sum_fuel                       => EDbio_vars%sum_fuel_patch             , & ! Output:
         fire_intensity                 => EDbio_vars%fire_intensity_patch       , & ! Output:
         fire_area                      => EDbio_vars%fire_area_patch            , & ! Output:
         scorch_height                  => EDbio_vars%scorch_height_patch        , & ! Output:
         fire_fuel_bulkd                => EDbio_vars%fire_fuel_bulkd_patch      , & ! Output:
         fire_fuel_eff_moist            => EDbio_vars%fire_fuel_eff_moist_patch  , & ! Output:
         fire_fuel_sav                  => EDbio_vars%fire_fuel_sav_patch        , & ! Output:
         fire_fuel_mef                  => EDbio_vars%fire_fuel_mef_patch        , & ! Output:
         litter_in                      => EDbio_vars%litter_in_patch            , & ! Output:
         litter_out                     => EDbio_vars%litter_out_patch           , & ! Output:
         seed_bank                      => EDbio_vars%seed_bank_patch            , & ! Output:
         seeds_in                       => EDbio_vars%seeds_in_patch             , & ! Output:
         seed_decay                     => EDbio_vars%seed_decay_patch           , & ! Output:
         seed_germination               => EDbio_vars%seed_germination_patch     , & ! Output:

         ED_biomass                     => EDbio_vars%ED_biomass_patch           , & ! InOut:
         ED_bdead                       => EDbio_vars%ED_bdead_patch             , & ! InOut:
         ED_bleaf                       => EDbio_vars%ED_bleaf_patch             , & ! InOut:
         ED_balive                      => EDbio_vars%ED_balive_patch            , & ! InOut:
         ED_bstore                      => EDbio_vars%ED_bstore_patch            , & ! InOut:
         phen_cd_status                 => EDbio_vars%phen_cd_status_patch       , & ! InOut:
 
         tlai                           => canopystate_vars%tlai_patch           , & ! InOut:
         elai                           => canopystate_vars%elai_patch           , & ! InOut:
         tsai                           => canopystate_vars%tsai_patch           , & ! InOut:
         esai                           => canopystate_vars%esai_patch           , & ! InOut:

         gpp                            => carbonflux_vars%gpp_patch             , & ! Output: 
         npp                            => carbonflux_vars%npp_patch               & ! Output:
         )

      ! ============================================================================
      ! Zero the whole variable so we dont have ghost values when patch number declines.
      ! ============================================================================

      trimming(:)             = 1.0_r8 !the default value of this is 1.0, making it 0.0 means that the output is confusing. 
      canopy_spread(:)        = 0.0_r8 
      PFTbiomass(:,:)         = 0.0_r8
      PFTleafbiomass(:,:)     = 0.0_r8 
      PFTstorebiomass(:,:)    = 0.0_r8
      PFTnindivs(:,:)         = 0.0_r8
      gpp(:)                  = 0.0_r8
      npp(:)                  = 0.0_r8  
      area_plant(:)           = 0.0_r8       
      area_trees(:)           = 0.0_r8        
      nesterov_fire_danger(:) = 0.0_r8
      spitfire_ROS(:)         = 0.0_r8
      effect_wspeed           = 0.0_r8
      TFC_ROS(:)              = 0.0_r8
      fire_intensity(:)       = 0.0_r8
      fire_area(:)            = 0.0_r8
      scorch_height(:)        = 0.0_r8
      fire_fuel_bulkd(:)      = 0.0_r8
      fire_fuel_eff_moist(:)  = 0.0_r8
      fire_fuel_sav(:)        = 0.0_r8
      fire_fuel_mef(:)        = 0.0_r8
      litter_in(:)            = 0.0_r8
      litter_out(:)           = 0.0_r8
      seed_bank(:)            = 0.0_r8          
      seeds_in(:)             = 0.0_r8
      seed_decay(:)           = 0.0_r8
      seed_germination(:)     = 0.0_r8
      ED_biomass(:)           = 0.0_r8
      ED_bdead(:)             = 0.0_r8
      ED_bleaf(:)             = 0.0_r8
      ED_bstore(:)            = 0.0_r8
      ED_balive(:)            = 0.0_r8
      phen_cd_status(:)       = 2

      do g = bounds%begg,bounds%endg
         currentSite => geds_local(g)%spnt
         if(firstsoilpatch(g) >= 0.and.currentSite%istheresoil == 1)then 

            ! ============================================================================
            ! Zero the bare ground tile BGC variables.
            ! ============================================================================

            trimming(firstsoilpatch(g))             = 1.0_r8 
            canopy_spread(firstsoilpatch(g))        = 0.0_r8 
            PFTbiomass(firstsoilpatch(g),:)         = 0.0_r8
            PFTleafbiomass(firstsoilpatch(g),:)     = 0.0_r8 
            PFTstorebiomass(firstsoilpatch(g),:)    = 0.0_r8
            PFTnindivs(firstsoilpatch(g),:)         = 0.0_r8
            gpp(firstsoilpatch(g))                  = 0.0_r8
            npp(firstsoilpatch(g))                  = 0.0_r8  
            area_plant(firstsoilpatch(g))           = 0.0_r8       
            area_trees(firstsoilpatch(g))           = 0.0_r8        
            nesterov_fire_danger(firstsoilpatch(g)) = 0.0_r8
            spitfire_ROS(firstsoilpatch(g))         = 0.0_r8
            TFC_ROS(firstsoilpatch(g))              = 0.0_r8
            effect_wspeed(firstsoilpatch(g))        = 0.0_r8
            fire_intensity(firstsoilpatch(g))       = 0.0_r8
            fire_area(firstsoilpatch(g))            = 0.0_r8
            scorch_height(firstsoilpatch(g))        = 0.0_r8
            fire_fuel_bulkd(firstsoilpatch(g))      = 0.0_r8
            fire_fuel_eff_moist(firstsoilpatch(g))  = 0.0_r8
            fire_fuel_sav(firstsoilpatch(g))        = 0.0_r8
            fire_fuel_mef(firstsoilpatch(g))        = 0.0_r8
            litter_in(firstsoilpatch(g))            = 0.0_r8
            litter_out(firstsoilpatch(g))           = 0.0_r8
            seed_bank(firstsoilpatch(g))            = 0.0_r8          
            seeds_in(firstsoilpatch(g))             = 0.0_r8
            seed_decay(firstsoilpatch(g))           = 0.0_r8
            seed_germination(firstsoilpatch(g))     = 0.0_r8
            ED_biomass(firstsoilpatch(g))           = 0.0_r8
            ED_balive(firstsoilpatch(g))            = 0.0_r8
            ED_bdead(firstsoilpatch(g))             = 0.0_r8
            ED_bstore(firstsoilpatch(g))            = 0.0_r8
            ED_bleaf(firstsoilpatch(g))             = 0.0_r8
            elai(firstsoilpatch(g))                 = 0.0_r8
            tlai(firstsoilpatch(g))                 = 0.0_r8
            tsai(firstsoilpatch(g))                 = 0.0_r8
            esai(firstsoilpatch(g))                 = 0.0_r8
            ED_bleaf(firstsoilpatch(g))             = 0.0_r8
            sum_fuel(firstsoilpatch(g))             = 0.0_r8
            !this should probably be site level. 
            phen_cd_status(firstsoilpatch(g))       = currentSite%status

            currentPatch => currentSite%oldest_patch
            do while(associated(currentPatch))

               if(currentPatch%patchno  <= numpft - numcft)then !don't expand into crop patches.   
                  p = currentPatch%clm_pno

                  currentCohort => currentPatch%shortest
                  do while(associated(currentCohort))
                     !accumulate into history variables. 
                     ft                    = currentCohort%pft
                     if(currentPatch%area.gt.0._r8)then
                        n_density             = currentCohort%n/currentPatch%area
                     else
                        n_density             = 0.0_r8
                     endif
                     ED_bleaf(p)           = ED_bleaf(p)           + n_density * currentCohort%bl 
                     ED_bstore(p)          = ED_bstore(p)          + n_density * currentCohort%bstore 
                     ED_biomass(p)         = ED_biomass(p)         + n_density * currentCohort%b 
                     ED_bdead(p)           = ED_bdead(p)           + n_density * currentCohort%bdead 
                     ED_balive(p)          = ED_balive(p)          + n_density * currentCohort%balive
                     npp(p)                = npp(p)                + n_density * currentCohort%npp 
                     gpp(p)                = gpp(p)                + n_density * currentCohort%gpp   
                     PFTbiomass(p,ft)      = PFTbiomass(p,ft)      + n_density * currentCohort%b
                     PFTleafbiomass(p,ft)  = PFTleafbiomass(p,ft)  + n_density * currentCohort%bl
                     PFTstorebiomass(p,ft) = PFTstorebiomass(p,ft) + n_density * currentCohort%bstore
                     PFTnindivs(p,ft)      = PFTnindivs(p,ft)      + currentCohort%n 
                     currentCohort         => currentCohort%taller
                  enddo ! cohort loop

                  !Patch specific variables that are already calculated

                  !These things are all duplicated. Should they all be converted to LL or array structures RF? 
                  nesterov_fire_danger(p) = currentSite%acc_NI 
                  spitfire_ROS(p)         = currentPatch%ROS_front 
                  TFC_ROS(p)              = currentPatch%TFC_ROS
                  effect_wspeed(p)        = currentPatch%effect_wspeed
                  fire_intensity(p)       = currentPatch%FI
                  fire_area(p)            = currentPatch%frac_burnt
                  scorch_height(p)        = currentPatch%SH
                  fire_fuel_bulkd(p)      = currentPatch%fuel_bulkd
                  fire_fuel_eff_moist(p)  = currentPatch%fuel_eff_moist
                  fire_fuel_sav(p)        = currentPatch%fuel_sav
                  fire_fuel_mef(p)        = currentPatch%fuel_mef                          
                  sum_fuel(p)             = currentPatch%sum_fuel                         
                  litter_in(p)            = sum(currentPatch%CWD_AG_in) +sum(currentPatch%leaf_litter_in)
                  litter_out(p)           = sum(currentPatch%CWD_AG_out)+sum(currentPatch%leaf_litter_out)
                  seed_bank(p)            = sum(currentPatch%seed_bank)
                  seeds_in(p)             = sum(currentPatch%seeds_in)
                  seed_decay(p)           = sum(currentPatch%seed_decay)
                  seed_germination(p)     = sum(currentPatch%seed_germination)
                  canopy_spread(p)        = currentPatch%spread(1) 
                  area_plant(p)           = currentPatch%total_canopy_area /currentPatch%area
                  area_trees(p)           = currentPatch%total_tree_area   /currentPatch%area
                  phen_cd_status(p)       = currentSite%status
                  if(associated(currentPatch%tallest))then
                     trimming(p)             = currentPatch%tallest%canopy_trim                
                  else
                     trimming(p) = 0.0_r8
                  endif

               else
                  write(iulog,*) 'ED: too many patches' 
               end if ! patchn<15

               currentPatch => currentPatch%younger
            end do !patch loop

         endif ! are there any soil patches?    
      enddo !gridcell loop

    end associate

  end subroutine update_ed_history_variables

  ! ============================================================================
  subroutine clm_indices( bounds, geds_local )

    implicit none

    type(bounds_type),                   intent(in)    :: bounds  ! clump bounds
    type(gridcell_edstate_type), target, intent(inout) :: geds_local( bounds%begg: )

    type (site) , pointer :: currentSite
    integer c,g
    integer, pointer :: cgridcell(:) 
    integer istheresoil(bounds%begg:bounds%endg) 
    integer landunit

    cgridcell => col%gridcell
    !decides whetehr this gridcell is subject to ED dynamics according to whether there is any soil landunit. 
    !not sure why this is a column loop. that seems extraneous to me. 
    istheresoil(bounds%begg:bounds%endg)  = 0
    do c = bounds%begc,bounds%endc
       g = cgridcell(c)   
       currentSite => geds_local(g)%spnt
       landunit = col%landunit(c)
       ! FIX(SPM,032414) check if istsoil is compatible with col%itype !
       ! make sure we only get the first patch value for places which have soil. 
       if(lun%itype(landunit) == istsoil.and.col%itype(c) == istsoil)then  
          istheresoil(g) = 1
       endif
       currentSite%istheresoil = istheresoil(g)
    enddo

  end subroutine clm_indices

  ! ============================================================================
  !          Load LAI in each layer into array to send to CLM
  ! ============================================================================
  subroutine clm_leaf_area_profile( cs_t, waterstate_vars, canopystate_vars )

    use EDGrowthFunctionsMod  , only: tree_lai, tree_sai, c_area 

    implicit none

    type (site)            , intent(inout),target :: cs_t
    type(waterstate_type)  , intent(inout)        :: waterstate_vars
    type(canopystate_type) , intent(inout)        :: canopystate_vars
    

    type (site)   , pointer :: currentSite
    type (patch)  , pointer :: currentPatch
    type (cohort) , pointer :: currentCohort

    real(r8) :: remainder !Thickness of layer at bottom of canopy. 
    real(r8) :: fleaf   ! fraction of cohort incepting area that is leaves.  
    integer  :: ft ! Plant functional type index. 
    integer  :: iv ! Vertical leaf layer index   
    integer  :: L  ! Canopy layer index
    integer  :: P  ! clm patch index  
    integer  :: C  ! column index

    real(r8) :: tlai_temp !calculation of tlai to check this method
    real(r8) :: elai_temp ! make a new elai based on the layer-by-layer snow coverage.
    real(r8) :: tsai_temp !
    real(r8) :: esai_temp !  

    real(r8) :: fraction_exposed         !how much of this layer is not covered by snow?
    real(r8) :: layer_top_hite           !notional top height of this canopy layer (m)
    real(r8) :: layer_bottom_hite        !notional bottom height of this canopy layer (m)
    integer  :: smooth_leaf_distribution !is the leaf distribution this option (1) or not (0)
    real(r8) :: frac_canopy(N_HITE_BINS) !amount of canopy in each height class
    real(r8) :: minh(N_HITE_BINS)        !minimum height in height class (m)
    real(r8) :: maxh(N_HITE_BINS)        !maximum height in height class (m)
    real(r8) :: dh                       !vertical detph of height class (m)
    real(r8) :: min_chite                !bottom of cohort canopy  (m)
    real(r8) :: max_chite                !top of cohort canopy      (m)
    real(r8) :: lai                      !summed lai for checking m2 m-2
    integer  :: NC                       !number of cohorts, for bug fixing. 

    smooth_leaf_distribution = 0

    associate(                                                                     & 
         snow_depth                 => waterstate_vars%snow_depth_col            , & !Input:
         frac_sno_eff               => waterstate_vars%frac_sno_eff_col          , & !Input: 
         snowdp                     => waterstate_vars%snowdp_col                , & !Output:

         frac_veg_nosno_alb         => canopystate_vars%frac_veg_nosno_alb_patch , & !Output:
         tlai                       => canopystate_vars%tlai_patch               , & !Output
         elai                       => canopystate_vars%elai_patch               , & !Output
         tsai                       => canopystate_vars%tsai_patch               , & !Output
         esai                       => canopystate_vars%esai_patch                 & !Output
         )

      !called from update_sites. 

      ! Here we are trying to generate a profile of leaf area, indexed by 'z' and by pft
      ! We assume that each point in the canopy recieved the light attenuated by the average
      ! leaf area index above it, irrespective of PFT identity... 
      ! Each leaf is defined by how deep in the canopy it is, in terms of LAI units.  (FIX(RF,032414), GB)
      
      currentSite => cs_t
      if(currentSite%istheresoil == 1)then
         currentPatch => currentSite%oldest_patch   
         p = currentPatch%clm_pno

         do while(associated(currentPatch))

            !Calculate tree and canopy areas. 
            currentPatch%canopy_area = 0._r8
            currentPatch%canopy_layer_lai(:) = 0._r8
            NC = 0
            currentCohort => currentPatch%shortest
            do while(associated(currentCohort))       
               currentCohort%c_area = c_area(currentCohort)
               currentPatch%canopy_area  = currentPatch%canopy_area + currentCohort%c_area
               NC = NC+1
               currentCohort => currentCohort%taller    
            enddo
            ! if plants take up all the tile, then so does the canopy.  
            currentPatch%canopy_area = min(currentPatch%canopy_area,currentPatch%area) 

            !calculate tree lai and sai.
            currentPatch%ncan(:,:) = 0 
            currentPatch%nrad(:,:) = 0 
            currentPatch%lai = 0._r8
            currentCohort => currentPatch%shortest
            do while(associated(currentCohort)) 
               currentCohort%treelai = tree_lai(currentCohort)    
               currentCohort%treesai = tree_sai(currentCohort)
               currentCohort%lai =  currentCohort%treelai *currentCohort%c_area/currentPatch%canopy_area 
               currentCohort%sai =  currentCohort%treesai *currentCohort%c_area/currentPatch%canopy_area  
               !Calculate the LAI plus SAI in each canopy storey. 
               currentCohort%NV =  CEILING((currentCohort%treelai+currentCohort%treesai)/dinc_ed)  

               currentPatch%ncan(currentCohort%canopy_layer,currentCohort%pft) = &
                    max(currentPatch%ncan(currentCohort%canopy_layer,currentCohort%pft),currentCohort%NV)
               currentPatch%lai = currentPatch%lai +currentCohort%lai

               do L = 1,nclmax-1
                  if(currentCohort%canopy_layer == L)then
                     currentPatch%canopy_layer_lai(L) = currentPatch%canopy_layer_lai(L) + currentCohort%lai + &
                          currentCohort%sai
                  endif
               enddo

               currentCohort => currentCohort%taller 

            enddo !currentCohort
            currentPatch%nrad = currentPatch%ncan

            if(smooth_leaf_distribution == 1)then
               ! we are going to ignore the concept of canopy layers, and put all of the leaf area into height banded bins. 
               ! using the same domains as we had before, except that CL always = 1
               currentPatch%tlai_profile = 0._r8
               currentPatch%tsai_profile = 0._r8  
               currentPatch%elai_profile = 0._r8
               currentPatch%esai_profile = 0._r8  

               ! this is a crude way of dividing up the bins. Should it be a function of actual maximum height? 
               dh = 1.0_r8*(HITEMAX/N_HITE_BINS) 
               do iv = 1,N_HITE_BINS  
                  if (iv == 1) then
                     minh(iv) = 0.0_r8
                     maxh(iv) = dh
                  else 
                     minh(iv) = (iv-1)*dh
                     maxh(iv) = (iv)*dh
                  endif
               enddo
               c = pft%column(currentPatch%clm_pno)
               currentCohort => currentPatch%shortest
               do while(associated(currentCohort))  
                  ft = currentCohort%pft
                  min_chite = currentCohort%hite - currentCohort%hite * EDecophyscon%crown(ft)
                  max_chite = currentCohort%hite  
                  do iv = 1,N_HITE_BINS  
                     frac_canopy(iv) = 0.0_r8
                     ! this layer is in the middle of the canopy
                     if(max_chite > maxh(iv).and.min_chite < minh(iv))then 
                        frac_canopy(iv)= min(1.0_r8,dh / (currentCohort%hite*EDecophyscon%crown(ft)))
                        ! this is the layer with the bottom of the canopy in it. 
                     elseif(min_chite < maxh(iv).and.min_chite > minh(iv).and.max_chite > maxh(iv))then 
                        frac_canopy(iv) = (maxh(iv) -min_chite ) / (currentCohort%hite*EDecophyscon%crown(ft))
                        ! this is the layer with the top of the canopy in it. 
                     elseif(max_chite > minh(iv).and.max_chite < maxh(iv).and.min_chite < minh(iv))then 
                        frac_canopy(iv) = (max_chite - minh(iv)) / (currentCohort%hite*EDecophyscon%crown(ft))
                     elseif(max_chite < maxh(iv).and.min_chite > minh(iv))then !the whole cohort is within this layer. 
                        frac_canopy(iv) = 1.0_r8
                     endif

                     ! no m2 of leaf per m2 of ground in each height class
                     currentPatch%tlai_profile(1,ft,iv) = currentPatch%tlai_profile(1,ft,iv) + frac_canopy(iv) * &
                          currentCohort%lai
                     currentPatch%tsai_profile(1,ft,iv) = currentPatch%tsai_profile(1,ft,iv) + frac_canopy(iv) * &
                          currentCohort%sai

                     !snow burial
                     fraction_exposed = 1.0_r8 !default. 

                     snowdp(c) = snow_depth(c) * frac_sno_eff(c)
                     if(snowdp(c) > maxh(iv))then
                        fraction_exposed = 0._r8
                     endif
                     if(snowdp(c) < minh(iv))then
                        fraction_exposed = 1._r8
                     endif
                     if(snowdp(c) >= minh(iv).and.snowdp(c) <= maxh(iv))then !only partly hidden... 
                        fraction_exposed =  max(0._r8,(min(1.0_r8,(snowdp(c)-minh(iv))/dh)))
                     endif

                     ! no m2 of leaf per m2 of ground in each height class
                     ! FIX(SPM,032414) these should be uncommented this and double check
                     !currentPatch%elai_profile(1,ft,iv) = currentPatch%tlai_profile(1,ft,iv) * fraction_exposed
                     !currentPatch%esai_profile(1,ft,iv) = currentPatch%tsai_profile(1,ft,iv) * fraction_exposed

                  enddo ! (iv) hite bins

                  currentCohort => currentCohort%taller

               enddo !currentCohort 

               !check
               currentPatch%lai = 0._r8
               currentCohort => currentPatch%shortest
               do while(associated(currentCohort)) 
                  currentPatch%lai = currentPatch%lai +currentCohort%lai
                  currentCohort => currentCohort%taller   
               enddo !currentCohort
               lai = 0.0_r8
               do ft = 1,numpft_ed
                  lai = lai+ sum(currentPatch%tlai_profile(1,ft,:))
               enddo

               if(lai > currentPatch%lai)then
                  write(iulog,*) 'problem with lai assignments'
               endif


            else ! smooth leaf distribution  
               !Go through all cohorts and add their leaf area and canopy area to the accumulators. 
               currentPatch%tlai_profile                = 0._r8
               currentPatch%tsai_profile                = 0._r8  
               currentPatch%elai_profile                = 0._r8
               currentPatch%esai_profile                = 0._r8  
               currentPatch%canopy_area_profile(:,:,:)  = 0._r8       
               currentPatch%ncan(:,:)                   = 0 
               currentPatch%nrad(:,:)                   = 0 
               currentCohort => currentPatch%shortest

               do while(associated(currentCohort))   
                  L = currentCohort%canopy_layer
                  ft = currentCohort%pft 
                  !Calculate the number of layers of thickness dlai, including the last one. 
                  currentCohort%NV =  CEILING((currentCohort%treelai+currentCohort%treesai)/dinc_ed)
                  !how much of each tree is stem area index? Assuming that there is 
                  if(currentCohort%treelai+currentCohort%treesai > 0._r8)then    
                     fleaf = currentCohort%lai / (currentCohort%lai + currentCohort%sai) 
                  else
                     fleaf = 0._r8
                     write(iulog,*) 'no stem or leaf area' ,currentCohort%pft,currentCohort%bl, &
                          currentCohort%balive,currentCohort%treelai,currentCohort%treesai,currentCohort%dbh, &
                          currentCohort%n,currentCohort%status_coh
                  endif
                  currentPatch%ncan(L,ft) = max(currentPatch%ncan(L,ft),currentCohort%NV)  
                  currentPatch%nrad(L,ft) = currentPatch%ncan(L,ft)  !fudge - this needs to be altered for snow burial
                  if(currentCohort%NV > currentPatch%nrad(L,ft))then
                     write(iulog,*) 'CF: issue with NV',currentCohort%NV,currentCohort%pft,currentCohort%canopy_layer
                  endif
                  c = pft%column(currentPatch%clm_pno)

                  !Whole layers.  Make a weighted average of the leaf area in each layer before dividing it by the total area. 
                  !fill up layer for whole layers.  FIX(RF,032414)- for debugging jan 2012
                  do iv = 1,currentCohort%NV-1 

                     currentPatch%tlai_profile(L,ft,iv) = currentPatch%tlai_profile(L,ft,iv)+ dinc_ed * fleaf * &
                          currentCohort%c_area/currentPatch%total_canopy_area
                     currentPatch%tsai_profile(L,ft,iv) = currentPatch%tsai_profile(L,ft,iv)+ dinc_ed * (1._r8 - fleaf) * &
                          currentCohort%c_area/currentPatch%total_canopy_area
                     currentPatch%canopy_area_profile(L,ft,iv) =  min(1.0_r8,currentPatch%canopy_area_profile(L,ft,iv) + &
                          currentCohort%c_area/currentPatch%total_canopy_area)

                     ! what is the height of this layer? (for snow burial purposes...)  
                     ! ecophyscon%vertical_canopy_frac(ft))! fudge - this should be pft specific but i cant get it to compile. 
                     layer_top_hite = currentCohort%hite-((iv/currentCohort%NV) * currentCohort%hite * &
                          EDecophyscon%crown(currentCohort%pft) )
                     layer_bottom_hite = currentCohort%hite-(((iv+1)/currentCohort%NV) * currentCohort%hite * &
                          EDecophyscon%crown(currentCohort%pft)) ! ecophyscon%vertical_canopy_frac(ft))
                     fraction_exposed = 1.0_r8 !default. 
                     snowdp(c) = snow_depth(c) * frac_sno_eff(c)
                     if(snowdp(c) > layer_top_hite)then
                        fraction_exposed = 0._r8
                     endif
                     if(snowdp(c) <= layer_bottom_hite)then
                        fraction_exposed = 1._r8
                     endif
                     if(snowdp(c) > layer_bottom_hite.and.snowdp(c) <= layer_top_hite)then !only partly hidden... 
                        fraction_exposed =  max(0._r8,(min(1.0_r8,(snowdp(c)-layer_bottom_hite)/ &
                             (layer_top_hite-layer_bottom_hite ))))
                     endif

                     currentPatch%elai_profile(L,ft,iv) =  currentPatch%tlai_profile(L,ft,iv) *fraction_exposed
                     !here we are assuming that the stem and leaf area indices have the same profile... 
                     currentPatch%esai_profile(L,ft,iv) =  currentPatch%tsai_profile(L,ft,iv) *fraction_exposed 
                  end do

                  !Bottom layer
                  iv = currentCohort%NV
                  ! ecophyscon%vertical_canopy_frac(ft))! fudge - this should be pft specific but i cant get it to compile.
                  layer_top_hite = currentCohort%hite-((iv/currentCohort%NV) * currentCohort%hite * &
                       EDecophyscon%crown(currentCohort%pft) )
                  ! ecophyscon%vertical_canopy_frac(ft))
                  layer_bottom_hite = currentCohort%hite-(((iv+1)/currentCohort%NV) * currentCohort%hite * &
                       EDecophyscon%crown(currentCohort%pft))
                  fraction_exposed = 1.0_r8 !default. 

                  fraction_exposed = 1.0_r8 !default. 
                  if(snowdp(c) > layer_top_hite)then
                     fraction_exposed = 0._r8
                  endif
                  if(snowdp(c) <= layer_bottom_hite)then
                     fraction_exposed = 1._r8
                  endif
                  if(snowdp(c) > layer_bottom_hite.and.snowdp(c) <= layer_top_hite)then !only partly hidden... 
                     fraction_exposed =  max(0._r8,(min(1.0_r8,(snowdp(c)-layer_bottom_hite) / &
                          (layer_top_hite-layer_bottom_hite ))))
                  endif

                  remainder = (currentCohort%treelai + currentCohort%treesai) - (dinc_ed*(currentCohort%NV-1))
                  if(remainder > 1.0_r8)then
                     write(iulog,*)'issue with remainder',currentCohort%treelai,currentCohort%treesai,dinc_ed, & 
                          currentCohort%NV
                  endif
                  !assumes that fleaf is unchanging FIX(RF,032414)

                  currentPatch%tlai_profile(L,ft,iv) =  currentPatch%tlai_profile(L,ft,iv)+ remainder * fleaf * &
                       currentCohort%c_area/currentPatch%total_canopy_area

                  !assumes that fleaf is unchanging FIX(RF,032414)

                  currentPatch%tsai_profile(L,ft,iv) =  currentPatch%tsai_profile(L,ft,iv)+  remainder * &
                       (1.0_r8-fleaf) * currentCohort%c_area/currentPatch%total_canopy_area
                  currentPatch%elai_profile(L,ft,iv) =  currentPatch%tlai_profile(L,ft,iv) *fraction_exposed
                  currentPatch%esai_profile(L,ft,iv) =  currentPatch%tsai_profile(L,ft,iv) *fraction_exposed
                  currentPatch%canopy_area_profile(L,ft,iv) = min(1.0_r8,currentPatch%canopy_area_profile(L,ft,iv) + &
                  currentCohort%c_area/currentPatch%total_canopy_area)

                  if(currentCohort%dbh <= 0._r8.or.currentCohort%n == 0._r8)then
                     write(iulog,*) 'ED: dbh or n is zero in clmedlink', currentCohort%dbh,currentCohort%n
                  endif
                  if(currentCohort%pft == 0.or.currentCohort%canopy_trim <= 0._r8)then
                     write(iulog,*) 'ED: PFT or trim is zero in clmedlink',currentCohort%pft,currentCohort%canopy_trim
                  endif
                  if(currentCohort%balive <= 0._r8.or.currentCohort%bl < 0._r8)then
                     write(iulog,*) 'ED: balive is zero in clmedlink',currentCohort%balive,currentCohort%bl
                  endif

                  currentCohort => currentCohort%taller

               enddo !cohort

               do L = 1,currentPatch%NCL_p
                  do ft = 1,numpft_ed
                     do iv = 1,currentPatch%nrad(L,ft)
                        !account for total canopy area
                        currentPatch%tlai_profile(L,ft,iv) = currentPatch%tlai_profile(L,ft,iv) / &
                             currentPatch%canopy_area_profile(L,ft,iv)
                        currentPatch%tsai_profile(L,ft,iv) = currentPatch%tsai_profile(L,ft,iv) / &
                             currentPatch%canopy_area_profile(L,ft,iv)
                        currentPatch%elai_profile(L,ft,iv) = currentPatch%elai_profile(L,ft,iv) / &
                             currentPatch%canopy_area_profile(L,ft,iv)
                        currentPatch%esai_profile(L,ft,iv) = currentPatch%esai_profile(L,ft,iv) / &
                             currentPatch%canopy_area_profile(L,ft,iv)
                     enddo

                     currentPatch%tlai_profile(L,ft,currentPatch%nrad(L,ft)+1: nlevcan_ed) = 0._r8
                     currentPatch%tsai_profile(L,ft,currentPatch%nrad(L,ft)+1: nlevcan_ed) = 0._r8
                     currentPatch%elai_profile(L,ft,currentPatch%nrad(L,ft)+1: nlevcan_ed) = 0._r8 
                     currentPatch%esai_profile(L,ft,currentPatch%nrad(L,ft)+1: nlevcan_ed) = 0._r8

                  enddo
               enddo

               !what is the resultant leaf area? 

               tlai_temp = 0._r8
               elai_temp = 0._r8
               tsai_temp = 0._r8
               esai_temp = 0._r8

               do L = 1,currentPatch%NCL_p
                  do ft = 1,numpft_ed

                     tlai_temp = tlai_temp + sum(currentPatch%canopy_area_profile(L,ft,1:currentPatch%nrad(L,ft)) * &
                          currentPatch%tlai_profile(L,ft,1:currentPatch%nrad(L,ft)))
                     elai_temp = elai_temp + sum(currentPatch%canopy_area_profile(L,ft,1:currentPatch%nrad(L,ft)) * &
                          currentPatch%elai_profile(L,ft,1:currentPatch%nrad(L,ft)))
                     tsai_temp = tsai_temp + sum(currentPatch%canopy_area_profile(L,ft,1:currentPatch%nrad(L,ft)) * &
                          currentPatch%tsai_profile(L,ft,1:currentPatch%nrad(L,ft)))
                     esai_temp = esai_temp + sum(currentPatch%canopy_area_profile(L,ft,1:currentPatch%nrad(L,ft)) * &
                          currentPatch%esai_profile(L,ft,1:currentPatch%nrad(L,ft)))
                  enddo
               enddo

               p = currentPatch%clm_pno
               if(abs(tlai(p)-tlai_temp) > 0.0001_r8) then

                  write(iulog,*) 'error with tlai calcs',&
                       NC,currentSite%clmgcell, abs(tlai(p)-tlai_temp), tlai_temp,tlai(p)

                  do L = 1,currentPatch%NCL_p
                     write(iulog,*) 'carea profile',L,currentPatch%canopy_area_profile(L,1,1:currentPatch%nrad(L,1))
                     write(iulog,*) 'tlai profile',L,currentPatch%tlai_profile(L,1,1:currentPatch%nrad(L,1))
                  end do

               endif

               elai(p) = max(0.1_r8,elai_temp)
               tlai(p) = max(0.1_r8,tlai_temp)
               esai(p) = max(0.1_r8,esai_temp)
               tsai(p) = max(0.1_r8,tsai_temp)

               ! write(iulog,*) 'elai',elai(p),tlai(p),tlai_temp,elai_temp
               ! write(iulog,*) 'esai',esai(p),tsai(p)
               !            write(iulog,*) 'TLAI_prof',currentPatch%tlai_profile(1,:,:)

               ! Fraction of vegetation free of snow. What does this do? Is it right? 
               if ((elai(p) + esai(p)) > 0._r8) then
                  frac_veg_nosno_alb(p) = 1_r8
               else
                  frac_veg_nosno_alb(p) = 0_r8
               end if
               ! write(iulog,*) 'frac nosno',frac_veg_nosno_alb(p)

               currentPatch%nrad = currentPatch%ncan
               do L = 1,currentPatch%NCL_p
                  do ft = 1,numpft_ed
                     if(currentPatch%nrad(L,ft) > 30)then
                        write(iulog,*) 'ED: issue w/ nrad'
                     endif
                     currentPatch%present(L,ft) = 0
                     do  iv = 1, currentPatch%nrad(L,ft);
                        if(currentPatch%canopy_area_profile(L,ft,iv) > 0._r8)then
                           currentPatch%present(L,ft) = 1     
                        endif
                     end do !iv
                  enddo !ft

                  if ( L .eq. 1 .and. abs(sum(currentPatch%canopy_area_profile(1,1:numpft_ed,1))) < 0.99999  &
                       .and. currentPatch%NCL_p .gt. 1 ) then
                          write(iulog,*) 'canopy area too small',sum(currentPatch%canopy_area_profile(1,1:numpft_ed,1))
                          write(iulog,*) 'cohort areas', currentPatch%canopy_area_profile(1,1:numpft_ed,:)
                  endif

                  if (L .eq. 1 .and. currentPatch%NCL_p .gt. 1 .and.  &
                       abs(sum(currentPatch%canopy_area_profile(1,1:numpft_ed,1))) < 0.99999) then
                          write(iulog,*) 'not enough area in the top canopy', &
                              sum(currentPatch%canopy_area_profile(L,1:numpft_ed,1)), &
                              currentPatch%canopy_area_profile(L,1:numpft_ed,1)
                  endif

                  if(abs(sum(currentPatch%canopy_area_profile(L,1:numpft_ed,1))) > 1.00001)then
                     write(iulog,*) 'canopy-area-profile wrong',sum(currentPatch%canopy_area_profile(L,1:numpft_ed,1)), &
                          
                          currentSite%clmgcell,currentPatch%patchno,L
                     write(iulog,*) 'areas',currentPatch%canopy_area_profile(L,1:2,1),currentPatch%patchno

                     currentCohort => currentPatch%shortest

                     do while(associated(currentCohort))

                     if(currentCohort%canopy_layer==1)then
                        write(iulog,*) 'cohorts',currentCohort%dbh,currentCohort%c_area, &
                             currentPatch%total_canopy_area,currentPatch%area,currentPatch%canopy_area
                        write(iulog,*) 'fracarea',currentCohort%pft, currentCohort%c_area/currentPatch%total_canopy_area
                     endif

                        currentCohort => currentCohort%taller  

                     enddo !currentCohort
                  endif
               enddo ! loop over L

               do L = 1,currentPatch%NCL_p
                  do ft = 1,numpft_ed
                     if(currentPatch%present(L,FT) > 1)then
                        write(iulog,*) 'present issue',currentPatch%clm_pno,L,ft,currentPatch%present(L,FT)
                        currentPatch%present(L,ft) = 1
                     endif
                  enddo
               enddo

            endif !leaf distribution

            currentPatch => currentPatch%younger 

         enddo !patch       
      endif !is there soil? 

    end associate 

  end subroutine clm_leaf_area_profile

! ============================================================================
end module EDCLMLinkMod
